<?xml version='1.0'?>
<!--
   Copyright 2006, Garrett A. Wollman
   All rights reserved.
  
   Redistribution and use in source and formatted forms, with or without
   modification, are permitted provided that the following conditions
   are met:
   1. Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.
   2. Redistributions in formatted form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.
  
   THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
   ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
   FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
   DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
   OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
   HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
   LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
   OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
   SUCH DAMAGE.
  
   $Id: make-metadata.xsl,v 1.9 2006/01/26 05:21:06 wollman Exp $
-->
<!--
   make-metadata.xsl: extract metadata from a photo gallery

   There are three kinds of metadata which may be found in a photo
   gallery index:

	1) Explicit metadata called out in the DTD, comprising the following:
	   - photos/title, photos/@copyright-year
	   - photo/title, photo/@date, photo/@format, photo/@photog,
	     photo/size[], photo/see-also
	   These have natural representations in RDF/XML; we use
	   the Dublin Core property set.

	2) Explicit metadata added by the user.  The content model of
	   the title and description elements is parsed character data
	   (#PCDATA), which allows any well-formed XML.  We assume
	   that any namespace-qualified elements found there are
	   metadata.  However, there is a small issue: both the
	   photograph itself and the description of the photograph may
	   have metadata.  We have resolved this conflict as follows:

		- Metadata found in the top-level description is
		  presumed to refer to the entire collection, and
		  references "/index.html" (or whatever the name of
		  the index file is configured to be).

		- Metadata found in a photo description is presumed
		  to refer to that description, and not necessarily to
		  the photo.

		- We have added a new, optional metadata element which
		  follows description in the photo content model.
		  Metadata found in a photo's metadata element refers
		  only to the photo, not to the description.

		- In all cases, metadata in the RDF namespace is
		  presumed to be a complete RDF statement.

	3) Metadata implicit in the structure of the photo gallery.
	   For example, all of the sizes of the images are alternate
	   versions of the same thing; the description page for each
	   image is (at least notionally) a description of the image.
-->
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
		xmlns:rdf="http://www.w3.org/1999/02/22-rdf-syntax-ns#"
		xmlns:dc="http://purl.org/dc/elements/1.1/"
		xmlns:dcterms="http://purl.org/dc/terms/"
		xmlns:rdfs="http://www.w3.org/2000/01/rdf-schema#"
		xmlns:foaf="http://xmlns.com/foaf/0.1/"
		xmlns:owl="http://www.w3.org/2002/07/owl#"
		xmlns:photo="http://www.holygoat.co.uk/owl/2005/05/photo/"
		xmlns:exifi="http://nwalsh.com/rdf/exif-intrinsic#"
		version='1.0'>
  <xsl:output method='xml' indent='yes'/>
  <xsl:param name="indexhtml"/>
  <xsl:param name="photogs"/>

  <xsl:template match="/">
    <xsl:comment>
      <xsl:text>Automatically generated: do not edit!</xsl:text>
    </xsl:comment>
    <rdf:RDF>
      <xsl:for-each select="document($photogs, current())/list/item">
        <foaf:Person rdf:nodeID="{@id}">
          <foaf:name>
            <xsl:value-of select="text()"/>
          </foaf:name>
        </foaf:Person>
      </xsl:for-each>
      <xsl:apply-templates/>
    </rdf:RDF>
  </xsl:template>

  <xsl:template match="/photos">
    <!-- need to do this first as some of the description metadata may
         reference global metadata -->
    <xsl:apply-templates select="metadata/*"/>
    <foaf:Document rdf:about="{$indexhtml}">
      <dc:title>
	<xsl:apply-templates select="title"/>
      </dc:title>
      <xsl:apply-templates select="description/*[namespace-uri() !=
	   'http://www.w3.org/1999/02/22-rdf-syntax-ns#' and
	   namespace-uri() != 'http://www.w3.org/1999/xhtml']"/>
      <dc:type rdf:resource="http://purl.org/dc/dcmitype/Collection"/>
      <dcterms:dateCopyrighted>
	<xsl:apply-templates select="@copyright-year"/>
      </dcterms:dateCopyrighted>
    </foaf:Document>
    <xsl:apply-templates select="description/*[namespace-uri() =
	'http://www.w3.org/1999/02/22-rdf-syntax-ns#']"/>
    <xsl:apply-templates select="photo"/>
  </xsl:template>

  <!-- Suppress all photos which are not set to be shown. -->
  <xsl:template match="photo[@show != 'yes']"/>

  <xsl:template match="photo[@show != 'no']">
    <!-- Image metadata first. -->
    <photo:Image rdf:nodeID="{concat('all-', @number, '-', @format)}">
      <dc:title>
	<xsl:apply-templates select="title"/>
      </dc:title>
      <xsl:if test="title/@short-form and title/@short-form/text() !=
		    title/text()">
	<dcterms:alternative>
	  <xsl:value-of select="title/@short-form"/>
	</dcterms:alternative>
      </xsl:if>
      <xsl:for-each select="size[id(@ref)/@show != 'no']">
	<dcterms:hasFormat rdf:resource="{concat(../@number, '-',
					  id(@ref)/@name, '.',
					  ../@format)}"/>
      </xsl:for-each>
      <dc:description rdf:nodeID="{concat('all-', @number, '-html')}"/>
      <dcterms:created>
	<xsl:value-of select="@date"/>
      </dcterms:created>
      <dc:creator rdf:nodeID="{document($photogs, current())/list/item[text() = current()/@photog]/@id}"/>
      <xsl:apply-templates select="metadata/*[namespace-uri() !=
	      'http://www.w3.org/1999/02/22-rdf-syntax-ns#']"/>
    </photo:Image>
    <xsl:apply-templates select="metadata/*[namespace-uri() =
	      'http://www.w3.org/1999/02/22-rdf-syntax-ns#']"/>

    <xsl:for-each select="size[id(@ref)/@show != 'no']">
      <photo:ImageFile rdf:about="{concat(../@number, '-',
                                     id(@ref)/@name, '.',
                                     ../@format)}">
        <xsl:if test="position() != 1">
          <foaf:thumbnail rdf:resource="{concat(../@number, '-', 
                                        id(../size[1]/@ref)/@name, '.',
                                        ../@format)}"/>
        </xsl:if>
        <xsl:choose>
          <xsl:when test="@orientation = 'landscape'">
            <exifi:width><xsl:value-of select="id(@ref)/@width"/></exifi:width>
            <exifi:height><xsl:value-of select="id(@ref)/@height"/></exifi:height>
          </xsl:when>
          <xsl:otherwise>
            <exifi:width><xsl:value-of select="id(@ref)/@height"/></exifi:width>
            <exifi:height><xsl:value-of select="id(@ref)/@width"/></exifi:height>
          </xsl:otherwise>
        </xsl:choose>
        <dc:description rdf:resource="{concat(../@number, '-', id(@ref)/@name,
                                      '.html')}"/>
        <photo:concreteRepresentationOf rdf:nodeID="{concat('all-', ../@number, '-', ../@format)}"/>
        <dcterms:isFormatOf rdf:nodeID="{concat('all-', ../@number, '-', ../@format)}"/>
        <dc:type rdf:resource="http://purl.org/dc/dcmitype/StillImage"/>
        <dc:format rdf:datatype="http://purl.org/dc/terms/IMT">
          <xsl:choose>
            <xsl:when test="../@format = 'jpg'">
              <xsl:text>image/jpeg</xsl:text>
            </xsl:when>
            <xsl:when test="../@format = 'tiff'">
              <xsl:text>image/tiff</xsl:text>
            </xsl:when>
            <xsl:when test="../@format = 'png'">
              <xsl:text>image/png</xsl:text>
            </xsl:when>
            <xsl:otherwise>
              <xsl:text>application/octet-stream</xsl:text>
            </xsl:otherwise>
          </xsl:choose>
        </dc:format>
      </photo:ImageFile>
    </xsl:for-each>

    <!-- Now handle the description pages.  -->
    <foaf:Document rdf:nodeID="{concat('all-', @number, '-html')}">
      <dc:type rdf:resource="http://purl.org/dc/dcmitype/Text"/>
      <dcterms:isPartOf rdf:resource="{$indexhtml}"/>
      <dc:title>
	<xsl:apply-templates select="title"/>
      </dc:title>
      <xsl:if test="title/@short-form and title/@short-form/text() !=
		    title/text()">
	<dcterms:alternative>
	  <xsl:value-of select="title/@short-form"/>
	</dcterms:alternative>
      </xsl:if>
      <xsl:for-each select="size[position() != 1 and id(@ref)/@show != 'no']">
	<dcterms:hasVersion rdf:resource="{concat(../@number, '-',
					    id(@ref)/@name, '.html')}"/>
      </xsl:for-each>
      <dcterms:requires rdf:nodeID="{concat('all-', @number, '-', @format)}"/>
      <xsl:apply-templates select="description/*[namespace-uri() !=
		'http://www.w3.org/1999/02/22-rdf-syntax-ns#' and
		namespace-uri() != 'http://www.w3.org/1999/xhtml']"/>
    </foaf:Document>
    <xsl:apply-templates select="description/*[namespace-uri() =
	      'http://www.w3.org/1999/02/22-rdf-syntax-ns#']"/>
  </xsl:template>

  <xsl:template match="description/text()">
  </xsl:template>

  <!-- Copy any item in /photos/description to the result set,
       preserving all children and attributes.  This is the identity
       transformation, and is taken from the XSLT specification,
       section 7.5.  -->
  <xsl:template match="description//*|description//@*">
    <xsl:copy>
      <xsl:apply-templates select="@*|node()"/>
    </xsl:copy>
  </xsl:template>
  <xsl:template match="metadata//*|metadata//@*">
    <xsl:copy>
      <xsl:apply-templates select="@*|node()"/>
    </xsl:copy>
  </xsl:template>

</xsl:stylesheet>
