#!/usr/bin/perl -w
#
# Copy photos from a DCIM-compliant removable medium to permanent storage,
# creating appropriate files and metadata for Web presentation using the
# xml-photo-galleries system.
#
#  Copyright 2002-2005, Garrett A. Wollman
#  All rights reserved.
# 
#  Redistribution and use in source and formatted forms, with or without
#  modification, are permitted provided that the following conditions
#  are met:
#  1. Redistributions of source code must retain the above copyright
#     notice, this list of conditions and the following disclaimer.
#  2. Redistributions in formatted form must reproduce the above copyright
#     notice, this list of conditions and the following disclaimer in the
#     documentation and/or other materials provided with the distribution.
# 
#  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
#  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
#  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
#  ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
#  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
#  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
#  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
#  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
#  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
#  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
#  SUCH DAMAGE.
# 
# $Id: copyphotos.pl,v 1.8 2006/04/30 18:26:50 wollman Exp $

use File::Slurp qw//;
use IO::Handle;
use IO::File;
use Image::Info qw//;
use Image::Magick;
use XML::Writer;

use vars qw($datadir @defsizes $fullname %sizes $viewer *TTY);
use strict;

open(TTY, '</dev/tty') or die "/dev/tty: $!\n";

%sizes = ('t' => [210, 140, 'th', 'thumb'],
	  's' => [720, 480, 'sm', 'small'],
	  'm' => [1152, 768, 'med', 'medium'],
	  'l' => [1536, 1024, 'lrg', 'large'],
	  'x' => [undef, undef, 'orig', 'orig']);
@defsizes = ('t', 's', 'm');
$datadir = '@datadir@';

sub n_or_y_p ($);
sub process_directory ($$$$$$);
sub prompt ($);
sub sizename ($$$);
sub write_header ($$$$$$$);
sub y_or_n_p ($);

sub main {
    my ($source, $dest) = @ARGV;
    my (@userdata) = getpwuid($<);

    # Assume BSD-type GECOS field....
    $fullname = (split /,/, $userdata[6])[0];

    my (@path) = split /:/, $ENV{PATH};
    $viewer = (grep { -x $_  } map { $_ . '/xv' } @path)[0];
    die "xv not found anywhere on \$PATH\n"
	unless defined $viewer;

    $| = 1;

    if (!defined($source || $dest)) {
	die "$0: insufficient arguments\nusage:\n\t$0 source-dir dest-dir\n";
    }

    if (!-d $source or !-r $source) {
	die "$0: $source: $!\n";
    }
    if (!-d $source . '/dcim') {
	die "$0: $source does not appear to be DCIM-compliant\n";
    }

    if (-e $dest and !-d $dest) {
	die "$0: $dest: not a directory\n";
    } elsif (!-e $dest) {
	my ($yn) = y_or_n_p("$dest does not exist.  Create?");
	if ($yn) {
	    mkdir($dest, 0777)
		or die "$0: $dest: $!\n";
	} else {
	    exit 2;
	}
    }

    my ($year) = (localtime(time))[5] + 1900;
    my ($copyright) = "Copyright $year, $fullname";
    print STDOUT ("Default copyright notice is \"$copyright\".\n");
    if (!y_or_n_p("Use default copyright notice?")) {
	print STDOUT ("Enter new copyright notice, or leave blank for no notice.\n");
	$copyright = <TTY>;
	undef $copyright if $copyright eq '';
    }

    print STDOUT ("Sizes are:\n",
		  (map { "\t$_: $sizes{$_}->[0]x$sizes{$_}->[1]\n" } 
		  (grep { $_ ne 'x' } keys %sizes)),
		  "\tx: original size (not shown on Web)\n");
    my (@sizes);
    while (1) {
	my ($sizes) = prompt ("Select sizes to generate [" 
			      . join("", @defsizes, 'x')
			      . "]");
	if ($sizes eq '') {
	    @sizes = (@defsizes, 'x');
	} else {
	    @sizes = grep { exists($sizes{$_}) } split (//, $sizes);
	}
	last if (@sizes);

	print STDOUT ("You'll have to do better than \"$sizes\"!\n");
    }

    my (@dirs) = grep { -d $_ } (map { $source . '/dcim/' . $_ } 
			         File::Slurp::read_dir($source . '/dcim'));
    die "$0: no digital image data found in ${source}/dcim\n"
	unless (@dirs);

    my ($index) = new IO::File(">$dest/index.xml")
	or die "$0: $dest/index.xml: $!\n";

    my ($w) = new XML::Writer(DATA_MODE => 1, DATA_INDENT  => 2,
			      OUTPUT => $index)
	or die "new XML::Writer: $!\n";

    foreach my $dir (@dirs) {
	next unless $dir =~ m,/dcim/(\d{3}),;
	process_directory ($dir, $dest, $1, $copyright, \@sizes, $w);
    }
    $w->endTag('photos');
    $w->end();
}

my $header_written = 0;

sub write_header ($$$$$$$) {
    my ($w, $sizes, $orig_w, $orig_h, $yyyy, $mm, $dd) = @_;

    return if $header_written;

    $w->xmlDecl('us-ascii');
    $w->doctype('photos',
		    "-//Boston Radio Archives//DTD Photo Gallery//EN",
		    "$datadir/photo-dtd.xml",
		    "<!ENTITY def.photog \"$fullname\">\
<!ENTITY def.format \"jpg\">\
<!ENTITY def.lang \"en-US\">\n");
    $w->startTag('photos', 
		 'date' => "$yyyy-$mm-$dd",
		 'copyright-year' => (localtime(time))[5] + 1900);
    $w->startTag('sizes');
    foreach my $size (grep { $_ ne 'x' } @$sizes) {
	$w->emptyTag('sizeinfo',
		     'id' => $sizes{$size}->[3],
		     'width' => $sizes{$size}->[0],
		     'height' => $sizes{$size}->[1],
		     'name' => $sizes{$size}->[2]);
    }
    if (grep { $_ eq 'x' } @$sizes) {
	$w->emptyTag('sizeinfo',
		     'id' => $sizes{'x'}->[3],
		     'width' => $orig_w,
		     'height' => $orig_h,
		     'name' => $sizes{'x'}->[2],
		     'show' => 'no');
    }
    $w->endTag('sizes');
    $w->startTag('title');
    $w->characters("\nEnter your title here\n");
    $w->endTag('title');
    $w->startTag('description');
    $w->cdata("\nintroduction goes here\n");
    $w->endTag('description');

    $header_written = 1;
}

sub y_or_n_p ($) {
    my ($prompt) = @_;
    my ($ans);

    $ans = prompt "$prompt [yn]";
    return ($ans !~ /^n/i);
}

sub n_or_y_p ($) {
    my ($prompt) = @_;
    my ($ans);

    $ans = prompt "$prompt [ny]";
    return ($ans =~ /^y/i);
}

sub prompt ($) {
    my ($prompt) = @_;
    my ($ans);

    print STDOUT "$prompt ";
    $ans = <TTY>;
    die "EOF on standard input\n" unless(defined $ans);
    chomp $ans;
    return $ans;
}

sub process_directory ($$$$$$) {
    my ($source, $dest, $group, $copyright, $sizes, $w) = @_;

    my (@files) = sort { $a cmp $b } (File::Slurp::read_dir($source));

file:
    foreach my $infile (@files) {
	my ($tag, $orig_copy, $image_data);

	if ($infile !~ /^(?:img_|dsc)(\d+)\.jpg$/) {
	    print STDERR ("Ignoring unknown file $infile\n");
	    next file;
	}

	$tag = $group . '-' . $1;
	$orig_copy = sizename($dest, $tag, 'x');

	print STDOUT ("Processing $tag....\n");

	eval {
	    $image_data = File::Slurp::read_file($source . '/' . $infile);
	    File::Slurp::write_file($orig_copy, $image_data);
	};
	if ($@) {
	    die "$0: $@\n";
	}

	local ($SIG{INT}) = sub { 
	    unlink $orig_copy; $SIG{INT} = 'DEFAULT'; kill('INT', $$);
	};

	#
	# Now that we have a copy of the file on fast storage, we can
	# figure out what it is and present it to the user for approval.
	# Image::Info reads the EXIF tags in the digital-camera files so
	# we know when the photograph was taken.
	#
	my ($info) = Image::Info::image_info(\$image_data);
	if (exists $info->{error}) {
	    warn "$0: unable to read $infile: $info->{error}\n";
	    next file;
	}
	my (@dim) = Image::Info::dim($info);
	print STDOUT ("Image dimensions: $dim[0]x$dim[1]\n");
	my ($yyyy, $mo, $dd, $hh, $mi, $ss) =
	    $info->{DateTimeOriginal} =~ /^(\d\d\d\d):(\d\d):(\d\d) (\d\d):(\d\d):(\d\d)$/;
	if (!defined $ss) {
	    warn "$0: unable to determine date of image $tag\n";
	} else {
	    print STDOUT ("Photograph taken on $yyyy-$mo-$dd at $hh:$mi:$ss\n");
	}

	#
	# At this point, we have enough information to write the file header.
	# (We do this now, rather than at program startup, so that we can
	# supply a useful value for the dimensions of an original image.)
	# write_header() makes certain to only write the header once.
	#
	write_header($w, $sizes, $dim[0], $dim[1], $yyyy, $mo, $dd);

	my ($exif_orientation);

	if ($info->{Orientation} eq 'left_bot') {
	    print STDOUT ("Orientation is left-handed portrait.\n");
	    $exif_orientation = -90;
	} elsif ($info->{Orientation} eq 'right_top') {
	    print STDOUT ("Orientation is right-handed portrait.\n");
	    $exif_orientation = 90;
	} elsif ($info->{Orientation} eq 'bot_right') {
	    print STDOUT ("Orientation is upside-down landscape.\n");
	    $exif_orientation = 180;
	} else {
	    print STDOUT ("Orientation is landscape.\n");
	    $exif_orientation = 0;
	}
	    
	#
	# XV and ImageMagick disagree as to the direction of angle measurement.
	#
	system ($viewer, qw(-geometry +0+0 -expand 0.5 -rotate), 
		-$exif_orientation, $orig_copy);

	my ($dorotate) = n_or_y_p("Rotate image?");
	if ($dorotate) {
	    #
	    # If we displayed the image rotated and the user said to
	    # rotate it, then assume we (or the camera) got it wrong.
	    #
	    $dorotate = $exif_orientation ? 0 : -90;
	} else {
	    #
	    # Otherwise, we got it right.
	    #
	    $dorotate = $exif_orientation;
	}

	my ($donormalize) = n_or_y_p("Normalize image?");
	my ($doshow) = y_or_n_p("Show this image?");
	my ($description) = prompt "Description:";

size:
	foreach my $size (@$sizes) {
	    next size if $size eq 'x';

	    my ($outfile) = sizename($dest, $tag, $size);

	    print STDOUT ("Creating $outfile...\n");
	    my ($magick) = new Image::Magick(magick => 'jpg');
	    my ($error) = $magick->BlobToImage($image_data);
	    if ("$error") {
		warn "$error";
		next file;
	    }
	    $magick->Normalize()
		if ($donormalize);
	    $magick->Resize(width => $sizes{$size}->[0],
			    height => $sizes{$size}->[1],
			    filter => 'lanczos',
			    support => 0.0);
	    if ($dorotate) {
		$magick->Rotate(degrees => $dorotate);
	    }
	    if (defined $copyright) {
		$magick->Comment($copyright);
	    }
	    $error = $magick->Write(filename => $outfile);
	    if ("$error") {
		warn "$0: $outfile: $error";
		next file;
	    }
	}
	unless (grep { $_ eq 'x' } @$sizes) {
	    unlink $orig_copy;
	}

	my (@attrib) = ('number' => $tag);
	if (!$doshow) {
	    push @attrib, ('show' => 'no');
	}
	if (defined $dd) {
	    push @attrib, ('date' => "$yyyy-$mo-$dd");
	}
	my ($infotxt) = '';

	if (defined $ss) {
	    $infotxt .= "Time: $hh:$mi:$ss\n";
	}
	if (exists $info->{Flash}) {
	    $infotxt .= "Flash: $info->{Flash}\n";
	}
	if (exists($info->{ExposureProgram}) and exists($info->{ExposureTime})) {
	    $infotxt .= "Exposure: $info->{ExposureProgram} $info->{ExposureTime}\n";
	}
	if (exists($info->{LightSource}) and $info->{LightSource} ne 'unknown') {
	    $infotxt .= "Light source: $info->{LightSource}\n";
	}
	$w->startTag('photo', @attrib);
	$w->comment("\n$infotxt")
	    unless $infotxt eq '';

	foreach my $size (@$sizes) {
	    $w->emptyTag('size', 'ref' => $sizes{$size}->[3],
			 'orientation' => ($dorotate ? 'portrait'
					   : 'landscape'));
	}
	$w->dataElement('processing', '-filter lanczos -support 0.0 '
			. ($donormalize ? "-normalize " : '')
			. ($dorotate != 0 ? "-rotate $dorotate" : ''));
	$w->dataElement('title', $description);
	$w->startTag('description');
	$w->cdata("\n$description\n");
	$w->endTag('description');
	$w->emptyTag('see-also', 'number' => '');
	$w->endTag('photo');
    }
}

sub sizename ($$$) {
    my ($dir, $tag, $size) = @_;

    return $dir . '/' . $tag . '-' . $sizes{$size}->[2] . '.jpg';
}

&main;
